/*!************************************************************************
 * FILE :         clAudioSourceController.cpp
 * SW-COMPONENT:  Audio Engine
 * DESCRIPTION:
 *
 * The main task of AudioSourceController is to perform source-switches
 * by activating and deactivating AudioSources.
 * It handles Priority checks by applying Stack-Rules, defined in clStackRules.cpp
 * AudioSources are kept in a stack, where the priority queue is realised.
 *
 * The typical straight forward approach for a source-switch is:
 *
 * - Client requests an AudioSource to be played
 * - Controller adds the requested AudioSource to the stack, applying
 *   priority rules etc.
 * - If the new AudioSource is on top of the stack, it has to be started
 * - Before, the previous active source has to be stopped.
 * - So the sequence is:
 *   - Start to mute the current active source and wait until the mute-ramp
 *   for this source has finished.
 *   - Then start the new source
 *
 * Things become a littel bit more complicated, when mix sources have to
 * be added, because the means, that more than on source can be active
 * at the same time.
 *
 * So the advanced approach is:
 * - Client requests an AudioSource to be played
 * - AudioSourceController stopps all sources, which should not be active
 * after applying the StackRules.
 * - Wait until all stopping source have finished the mute-ramp.
 * - AudioSourceController starts all sources, which should be active after
 * applying the StackRules.
 *
 *
 *
 * AUTHOR:        CM-DI/PJ-VW34 Steinle
 * COPYRIGHT:     (c) 2006 Blaupunkt Werke
 *************************************************************************/
#include "AudioStack/clAudioSourceController.h"
#include "AudioStack/AudioSources/clAudioSourceFactory.h"

//#include "GUI_Widget/Trace/GUI_Trace.h"

//#include "Audio_SM/Audio_SM_Trace_defines.h"
#include "AudioStack/clGeniviAudioCtrlAdapter.h"
#include "AudioStack/cliAudioSourceControllerObserver.h"
#include "AudioStack/SMT/clSrcStateFactory.h"
#include "AudioStack/AudioSources.h"

//#include "hsi/clHSIMngr.h"
//#include "hsi/clCCAhandlerGeneral.h"
//#include "fiSPM/clSPM_CCAhandler.h"

#include "AudioStack/clAudioSMEngine.h"

#ifndef USE_DLT_TRACE
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_COMP_AUDIOSTACK
#include "trcGenProj/Header/clAudioSourceController.cpp.trc.h"
#endif

namespace AudioStack
{

//Init static Members
clStack*                          clAudioSourceController::m_pcoAudioStack    = NULL;
clIAudioSourceControllerObserver* clAudioSourceController::m_poObserver       = NULL;
clAudioSourceController*          clAudioSourceController::m_poSelf_Singleton = NULL;

clAudioSourceController& clAudioSourceController::getInstance()
{
   if(m_poSelf_Singleton == NULL)
   {
      m_poSelf_Singleton = OSAL_NEW clAudioSourceController();
   }
   return *m_poSelf_Singleton;
}

/*!************************************************************************
 * METHOD:        constructor
 * CLASS:         clAudioSourceController
 * DESCRIPTION:   constructor
 *                Creates a new AudioStack and initializes it with the
 *                NONE Source. The NONE Source is a permanent dummy background
 *                source which makes most Stack operations easier.
 * PARAMETER:
 * RETURNVALUE:
 * AUTHOR:      (CM-DI/PJ-VW34)
 *************************************************************************/
clAudioSourceController::clAudioSourceController()
{
   m_poObserver = NULL;
   m_pcoAudioStack= OSAL_NEW clStack();
}


/*!************************************************************************
 * METHOD:        bSELECT
 * CLASS:         clAudioSourceController
 * DESCRIPTION:   Use this Interface Method to select the background source
 *                Backround sources (Entertainment Sources) are not stopped/removed.
 *                They are always exchanged/replaced by another one.
 *                Sequence is:
 *                - Stop or Pause current background source (mute-ramp)
 *                - Wait for asynchronous confirmation of stop(mute-ramp finished)
 *                - Send Broadcast Notification for source stopped
 *                - Send Broadcast Notification for Exchanged BG-Source
 *                - Start new background source (demute-ramp)
 *                - Wait for asynchronous confirmation of start(demute-ramp finished)
 *                - Send Broadcast Notification for source started
 *
 * PARAMETER:     u8newBgSource
 *                userData          reserved
 * RETURNVALUE:   bSucess
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tBool clAudioSourceController::bSELECT(SourceID srcID, tU32 u32UserData)
{
   tBool bResult = FALSE;
   clAudioSource* poNewBgSrc = NULL;
   clAudioSource* poUnderlyingBgSrc = NULL;
   clAudioSource* poCurrentActiveSource = NULL;

   poNewBgSrc = clAudioSourceFactory::getAudioSource(srcID);

   if (poNewBgSrc != NULL && m_pcoAudioStack != NULL)
   {
      m_pcoAudioStack->trace();

      poNewBgSrc->vSetObserver(this);
      // pass userData to AudioSource for use as soon as Source is called.
      // Currently used for Preset-Info only. (AM and FM sources)
      poNewBgSrc->vSetUserData(u32UserData);
      bResult = m_pcoAudioStack->AddSource(*poNewBgSrc);
      m_pcoAudioStack->trace();
      poUnderlyingBgSrc = m_pcoAudioStack->GetUnderlayingSource(*poNewBgSrc);


      // step 1: stop and remove prev. bg-source
      if (poUnderlyingBgSrc != NULL)
      {
         // different behaviour for 2 cases:
         // 1: underlying bg-source is active (rampUp, RampDownToPause, RampDownToOff, On)
         // 2:  underlying bg-source is NOT active (Pause, Off)

         if (poUnderlyingBgSrc->bIsActive() )
         {
          ETG_TRACE_USR4(("bSELECT, poUnderlyingBgSrc->bIsActive() is true "));
            // case 1: underlying bg is active:
            // stop it and send it to background

            // determine the bg-source, which is activated next in order to define the type of mute-ramp
            // and to indicate, which band to bring to Foreground
            poUnderlyingBgSrc->vOff(poNewBgSrc->sGetId());
         }
         else
         {
          ETG_TRACE_USR4(("bSELECT, poUnderlyingBgSrc->bIsActive() is false "));
            // case 2: underlying bg is NOT active:
            if (poUnderlyingBgSrc->pclGetState() == clSrcStateFactory::pclCreatePause())
            {
              ETG_TRACE_USR4(("bSELECT, underlying bg is NOT active "));
               // case 2.1: underlying src is in PAUSE
               // send it to Background by changing from pause to off

               // determine the bg-source, which is activated next in order to define the type of mute-ramp
               // and to indicate, which band to bring to Foreground
               poUnderlyingBgSrc->vOff(poNewBgSrc->sGetId());
            }
            else
            {
              ETG_TRACE_USR4(("bSELECT, underlying src is in OFF "));
               // case 2.2: underlying src is in OFF
               // example:
               //         |  newBG    | off
               //         |  BG1      | off
               //         |  BG0      | rampDownToOff
               m_pcoAudioStack->RemoveSource(*poUnderlyingBgSrc);
            }
         }
      }
      else
      {
        ETG_TRACE_USR4(("bSELECT, for preset only "));
         // for preset only (user defined action)
         // there is no underlying bg-source. So the requested source might be already on
         // stack in pause state
         if (poNewBgSrc->pclGetState() == clSrcStateFactory::pclCreateRampDownToPause()
             || poNewBgSrc->pclGetState() == clSrcStateFactory::pclCreatePause())
         {
          ETG_TRACE_USR4(("bSELECT, before poNewBgSrc->vPause "));
            poNewBgSrc->vPause(poNewBgSrc->sGetId());
         }
      }

      // step 2: start this new bg-source if required. Otherwise keep it in Pause
      // start the bg-source, if a) it is top of stack AND b) no other source is active
      poCurrentActiveSource = pcoGetCurrentActiveAudiosource();
      if (poNewBgSrc == pcoGetTopOfStackAudiosource()
         &&
         (poCurrentActiveSource == NULL || poCurrentActiveSource == poNewBgSrc))
      {
        //Check for PushAction WaitFor rule
        if(WaitForSourceNeeded(*poNewBgSrc))
        {
          ETG_TRACE_USR4(("bSELECT, before poNewBgSrc->vWaitFor "));
          poNewBgSrc->vWaitFor();
        }else{
           //We can proceed directly
          ETG_TRACE_USR4(("bSELECT, before poNewBgSrc->vOn "));
          poNewBgSrc->vOn();
        }
      }
      // if this new bg src is exchanged silent (an FG-source is on stack) and currently is in rampDownToOff
      // then change it from rampDownToOff to rampDownToPause to keep it in Foreground
      // Don't wait in this case until the prev. bg-source is stopped cause there is no stop in progress
      // example:
      //         |  FG       | off
      //         |  BG1      | off
      //         |  BG0      | rampDownToOff
      // -> SELECT(BG0)
      //         |  FG       | off
      //         |  BG0      | rampDownToOff
      //         |  BG1      | off
      else if (poNewBgSrc->pclGetState() == clSrcStateFactory::pclCreateRampDownToOff())
      {
        ETG_TRACE_USR4(("bSELECT, pclGetState is pclCreateRampDownToOff "));
         poNewBgSrc->vPause(poNewBgSrc->sGetId());
         //Now we might Kick the Top FG Src
         //Check if Kick Action is provided
         if(poNewBgSrc->getPushAction(0) == ((clStackRules::actions_t)((poCurrentActiveSource->sGetId().enSourceClass) | clStackRules::/*actions_t::*/action_abort)))
         {
             ETG_TRACE_USR1((" Aborting SourceClass: %d subID %#x(%d)"
               ,ETG_CENUM(AudioSources::enAudioSources,static_cast<AudioSources::enAudioSources>(poCurrentActiveSource->sGetId().enSourceClass))
               , (tU16)poCurrentActiveSource->sGetId().u16SubSource
               , (tU16)poCurrentActiveSource->sGetId().u16SubSource));
         }
      }else if( NULL  != pcoGetCurrentActiveAudiosource())
      {
        ETG_TRACE_USR4(("bSELECT,special-> vvd1"));
        m_pcoAudioStack->trace();
        clAudioSource* poActSrc = pcoGetCurrentActiveAudiosource();
                //there is an exclusive source on stack which is not Mix source
      ETG_TRACE_USR4(("bSELECT, there is an exclusive source on stack which is not Mix source "));
      ETG_TRACE_USR1((" pcoGetCurrentActiveAudiosource SourceClass: %d subID %#x(%d)"
               ,ETG_CENUM(AudioSources::enAudioSources,static_cast<AudioSources::enAudioSources>(poCurrentActiveSource->sGetId().enSourceClass))
               , (tU16)poCurrentActiveSource->sGetId().u16SubSource
               , (tU16)poCurrentActiveSource->sGetId().u16SubSource));

      /* case 1 : SYS_MUTE (State=on)
       *          -------------------
       *          MEDIA/FM (State=off)
       * */
      if((true == poActSrc->bIsActive()) &&
          (clStackRules::groupInternal == poActSrc->getGroup()))
      {
       ETG_TRACE_USR4(("bSELECT,found ActiveSrc of clStackRules::groupInternal,vPause of SourceClass: %d subID %#x(%d)"
               ,ETG_CENUM(AudioSources::enAudioSources,static_cast<AudioSources::enAudioSources>(poCurrentActiveSource->sGetId().enSourceClass))
               , (tU16)poCurrentActiveSource->sGetId().u16SubSource
               , (tU16)poCurrentActiveSource->sGetId().u16SubSource));
       //FM failed to Pause from Off state, issue in FM
       poNewBgSrc->vMW_Pause(poActSrc->sGetId());
      }
      /* case 3 : NAVI (State=on)
       *          -------------------
       *          PHONE (State=on)
       *          -------------------
       *          MEDIA/FM (State=pause)
       * */
      else if((true == poActSrc->bIsActive()) && (poActSrc->getType() == clStackRules::typeFg))
      {
        ETG_TRACE_USR4(("bSELECT,special-> Current active source is FG source, vPause of SourceClass: %d subID %#x(%d)"
               ,ETG_CENUM(AudioSources::enAudioSources,static_cast<AudioSources::enAudioSources>(poCurrentActiveSource->sGetId().enSourceClass))
               , (tU16)poCurrentActiveSource->sGetId().u16SubSource
               , (tU16)poCurrentActiveSource->sGetId().u16SubSource));
        poNewBgSrc->vMW_Pause(poActSrc->sGetId());
      }

      }
      else{
        ETG_TRACE_USR4(("bSELECT, else "));
      }

      // trace
      m_pcoAudioStack->trace();
   }
   return bResult;
}


/*!************************************************************************
 * METHOD:        bON
 * CLASS:         clAudioSourceController
 * DESCRIPTION:   Use this Interface Method to activate a Foreground Source
 *                A Foreground Source is added to the stack. The positon in
 *                the stack depends on the priority rules.
 *                By default an FG source does not replace another source.
 *                But keep in mind, that the stack rules might remove other
 *                Sources automatically.
 *                Sequence is:
 *                - Stop or Pause the sources below in the stack (mute-ramp)
 *                - Wait for asynchronous confirmation of stop(mute-ramp finished)
 *                - Send Broadcast Notification for sources, which have been stopped
 *                - Start this source, if it is the top-stack-source (demute-ramp)
 *                - Wait for asynchronous confirmation of start(demute-ramp finished)
 *                - Send Broadcast Notification for source started
 *                This method initiates this sequence by calling vMuteStack()
 *
 *
 * PARAMETER:     u8newFgSource
 *                userData          reserved
 * RETURNVALUE:   bSucess
 * AUTHOR:        (CM-DI/PJ-VW34)
 * CHANGES:
 *    22.01.2008 J. Steinle: Before: Mute-Demute-Phase was started only
 *               if  the new FG source was in mute-state. This was done
 *               to prevent mute in case that the source was already active.
 *               In case of toggling On-Off-Requests, an On-Request could be
 *               lost because source was in ramp-down-state and not in
 *               mute-state. Solution: Start mute-demute-ramp always.
 *               To prevent a mute in case that the source is already active
 *               the mute-ramp begins on the underlying source now.
 *************************************************************************/
tBool clAudioSourceController::bON(SourceID srcID, tU32 u32UserData)
{
   tBool bResult = FALSE;
   clAudioSource* poNewFgSrc = NULL;
   clAudioSource* poPrevActiveSrc = NULL;

   poNewFgSrc = clAudioSourceFactory::getAudioSource(srcID);
   poPrevActiveSrc = pcoGetCurrentActiveAudiosource();

   if (poNewFgSrc != NULL && m_pcoAudioStack != NULL)
   {
      //trace("bON - newSrc: %s", poNewFgSrc->pacGetName());
//      vTrace(AUD_SM_SRC_ON,poNewFgSrc->u8GetId());
      ETG_TRACE_USR4(("bON SourceClass: %d subID %#x(%d)"
            ,ETG_CENUM(AudioSources::enAudioSources,static_cast<AudioSources::enAudioSources>(poNewFgSrc->sGetId().enSourceClass))
            , poNewFgSrc->sGetId().u16SubSource
            , poNewFgSrc->sGetId().u16SubSource));
      // push new source on stack
      poNewFgSrc->vSetObserver(this);
      poNewFgSrc->vSetUserData(u32UserData);
      bResult = m_pcoAudioStack->AddSource(*poNewFgSrc);

      if (bResult)
      {
         // different behavior for 2 cases:
         // 1: the new FG source is the top of stack source
         // 2: the new FG source is NOT the top of stack source

         if (pcoGetTopOfStackAudiosource() == poNewFgSrc)
         {
            // case 1: the new FG source is top of stack source
            // start a mute-phase for the currently active source
            if (poPrevActiveSrc!= NULL && poPrevActiveSrc != poNewFgSrc)
            {
               // case 1.1: Another source is currently active. So start mute-phase for it.
               // only send pause to active source if not already in ramp down or off or pause
               // note: forbidden to send pause while a source is going to off, because
               // this would change the target state of that source
               if (poPrevActiveSrc->pclGetState() == clSrcStateFactory::pclCreateRampUp()
                   || poPrevActiveSrc->pclGetState() == clSrcStateFactory::pclCreateRampUpInit()
                   || poPrevActiveSrc->pclGetState() == clSrcStateFactory::pclCreateSrcAvailableCheck_On()
                   || poPrevActiveSrc->pclGetState() == clSrcStateFactory::pclCreateRampUpRequesting()
                   || poPrevActiveSrc->pclGetState() == clSrcStateFactory::pclCreateOn())
               {
                  // reset user data
                  poPrevActiveSrc->vSetUserData(0);
                  poPrevActiveSrc->vPause( poNewFgSrc->sGetId());
                  //poPrevActiveSrc->vOff(poNewFgSrc->u8GetId());
               }
            }
            else
            {
               // case 1.2: Either NO source is active (can occur at startup) or this FG is already active
               // send On-request to cancel a ramp-down for this new FG source
               // Check for PushAction WaitFor rule
               if(WaitForSourceNeeded(*poNewFgSrc))
               {
                 poNewFgSrc->vWaitFor();
               }else{
                 //We can proceed directly
                 poNewFgSrc->vOn();
               }
            }
         }
         else
         {
            // case 2: the new FG source is NOT the top of stack source
            // request for an FG-source which is not on top of stack, but somewhere below/between
            // example:
            //         |  FG0    | on                     |  FG0    | on
            //         |  newFG  | rampDownToOff     ->   |  newFG  | rampDownToPause
            //         |  BG     | pause                  |  BG     | pause
            if (poNewFgSrc->pclGetState() == clSrcStateFactory::pclCreateRampDownToOff())
            {
               poNewFgSrc->vSetUserData(0);
               poNewFgSrc->vPause(poNewFgSrc->sGetId());
            }
         }
      }

      // trace
      m_pcoAudioStack->trace();
   }
   else
   {
      ETG_TRACE_FATAL(("poNewFgSrc == NULL || m_pcoAudioStack == NULL"));
      bResult = FALSE;
   }
   return bResult;
}

/*!************************************************************************
 * METHOD:        bMIX
 * CLASS:         clAudioSourceController
 * DESCRIPTION:   Use this Interface Method to activate a Mix Source
 *                A Mix Source is added to the stack. The position in
 *                the stack depends on the priority rules.
 *                By default a Mix source does not replace another source.
 *                But keep in mind, that the stack rules might remove other
 *                Sources automatically.
 *                In case of Mix souces there is no need to mute other
 *                existing sources.
 *                Sequence is:
 *                - Start this source, if it is the top-stack-source (demute-ramp)
 *                - Wait for asynchronous confirmation of start(demute-ramp finished)
 *                - Send Broadcast Notification for source started
 *                This method initiates this sequence by calling vOn()
 *
 * PARAMETER:     u8newMixSource
 *                u32UserData          reserved
 * RETURNVALUE:   bSucess
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tBool clAudioSourceController::bMIX(SourceID srcID, tU32 u32UserData)
{
  tBool bResult = FALSE;
  clAudioSource* poNewMixSrc = NULL;

  poNewMixSrc = clAudioSourceFactory::getAudioSource(srcID);

  if (poNewMixSrc != NULL && m_pcoAudioStack != NULL)
  {
    ETG_TRACE_USR2(("bMIX SourceClass: %d subID %#x(%d)"
            ,ETG_CENUM(AudioSources::enAudioSources,static_cast<AudioSources::enAudioSources>(poNewMixSrc->sGetId().enSourceClass))
            , poNewMixSrc->sGetId().u16SubSource
            , poNewMixSrc->sGetId().u16SubSource));
    // push new source on stack
    poNewMixSrc->vSetObserver(this);
    poNewMixSrc->vSetUserData(u32UserData); // needed to support new acr_fi, providing a category of the announcement
    bResult = m_pcoAudioStack->AddSource(*poNewMixSrc);
    if (bResult)
    {
      //Check for PushAction WaitFor rule
      if (WaitForSourceNeeded(*poNewMixSrc))
      {
        poNewMixSrc->vWaitFor();
      }
      else
      {
        //We can proceed directly
        poNewMixSrc->vOn();
      }
      // a mix source can directly be started. The mute-demute-phases are skipped.
      // No Crossover and no BG-Exchange-Triggers are sent
    }
    // trace
    //m_pcoAudioStack->trace();
  }
  else
  {
    ETG_TRACE_ERR(("poNewMixSrc == NULL or m_pcoAudioStack == NULL"));
    bResult = FALSE;
  }
  return bResult;
}

tBool clAudioSourceController::WaitForSourceNeeded(clAudioSource &src)
{
  int i;
  clStackRules::actions_t action;
  //1. Check if Source defines a WaitFor rule
  for(i=0; (action=src.getPushAction(i)) != clStackRules::/*actions_t::*/end; i++)
  {
      switch(action & clStackRules::/*actions_t::*/mask)
      {
      case clStackRules::/*actions_t::*/action_waitFor:
      {
          clAudioSource* pTargetSrc =
                m_pcoAudioStack->GetTopSource(static_cast<AudioSources::enAudioSources>(action & clStackRules::/*actions_t::*/idxMask));
          if(pTargetSrc!=NULL && pTargetSrc->bIsActive())
          {
            ETG_TRACE_USR4(("WaitFor rule found: SourceClass: %d subID %#x(%d) WAITS FOR SourceClass: %d subID %#x(%d)"
            ,ETG_CENUM(AudioSources::enAudioSources,static_cast<AudioSources::enAudioSources>(src.sGetId().enSourceClass))
            , src.sGetId().u16SubSource
            , src.sGetId().u16SubSource
            ,ETG_CENUM(AudioSources::enAudioSources,static_cast<AudioSources::enAudioSources>(pTargetSrc->sGetId().enSourceClass))
            , pTargetSrc->sGetId().u16SubSource
            , pTargetSrc->sGetId().u16SubSource));
            //1.1.1 WaitFor Source, return false
            return TRUE;
          }else{
             if(pTargetSrc != NULL)
             {
                ETG_TRACE_USR4(("WaitFor rule found: Found NOT active SourceClass: %d subID %#x(%d)"
                  ,ETG_CENUM(AudioSources::enAudioSources,static_cast<AudioSources::enAudioSources>(pTargetSrc->sGetId().enSourceClass))
                  , pTargetSrc->sGetId().u16SubSource
                  , pTargetSrc->sGetId().u16SubSource));
             }else{
                ETG_TRACE_USR4(("WaitFor rule found: No conflicting SourceClass active"));
             }
          }
      }
      break;
      default:
        //do nothing
      break;
      }
  }
  //No WaitFor rule or no corresponding active Source
  return FALSE;
}


/*!************************************************************************
 * METHOD:        bOFF
 * CLASS:         clAudioSourceController
 * DESCRIPTION:   Use this Interface Method to deativate an FG Source
 *                Sequence is:
 *                - Stop or Pause this source (mute-ramp)
 *                - Wait for asynchronous confirmation of stop(mute-ramp finished)
 *                - Send Broadcast Notification (Source stopped)
 *                - Start the new top-stack-source (demute-ramp)
 *                - Wait for asynchronous confirmation of start(demute-ramp finished)
 *                - Send Broadcast Notification for source started
 *                This method initiates this sequence by calling vMuteStack()
 *
 *
 * PARAMETER:     u8FgSource
 *                u32UserData          reserved
 * RETURNVALUE:   bSucess
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tBool clAudioSourceController::bOFF(SourceID srcID, tU32 u32UserData)
{
   tBool bResult = FALSE;
   clAudioSource* poFgSrc = NULL;
   clAudioSource* poNextSource = NULL;

   poFgSrc = clAudioSourceFactory::getAudioSource(srcID);

   if (NULL != poFgSrc && m_pcoAudioStack != NULL)
   {
      //trace("bOFF(%s)", poFgSrc->pacGetName());
     ETG_TRACE_USR2(("bOFF SourceClass: %d subID %#x(%d)"
            ,ETG_CENUM(AudioStack::AudioSources::enAudioSources,static_cast<AudioSources::enAudioSources>(poFgSrc->sGetId().enSourceClass))
            , poFgSrc->sGetId().u16SubSource
            , poFgSrc->sGetId().u16SubSource));


      // different behaviour for 2 cases:
      // 1: the FG source is OFF (it's on stack but not yet started)
      // 2: the FG source is NOT off (pause, rampUp, rampDownToPause, rampDownToOff, on)

      if (poFgSrc->pclGetState() == clSrcStateFactory::pclCreateOff())
      {
         // case 1: FG is in OFF state
         // if this FG is the TopSource than cancel the RampDownToPause of the underlying src
         // Don't cancel a RampDownToOff of the underlying src
         m_pcoAudioStack->RemoveSource(*poFgSrc);
         poNextSource = pcoGetTopOfStackAudiosource();

         if (poNextSource != NULL
             && poNextSource->pclGetState() == clSrcStateFactory::pclCreateRampDownToPause() )
         {
            poNextSource->vOn();
         }
      }
      else
      {
         // case 2: FG is NOT in OFF state
         // in this case stop it and proceed later asynchronously on off_done()
         // find the next source to start and specify it in the Off-Command
         // Skip virtual sources (e.g. mute-sources) because FC-Audio needs a real source to set
         poNextSource = pcoGetTopOfStackAudiosource();
         while(poNextSource != NULL
            && (poNextSource == poFgSrc // the stopped src itself is topOfStack, so find the next below
               //|| poNextSource->getGroup() == clStackRules::groupInternal
               )) // the found src is not a real src, so find the next below
         {
            poNextSource = m_pcoAudioStack->GetUnderlayingSource(*poNextSource);
         }
         if (poNextSource != NULL)
         {
            ETG_TRACE_USR4(("Call vOFF of SourceClass: %d subID %#x(%d), PNS %d"
                        ,ETG_CENUM(AudioStack::AudioSources::enAudioSources,static_cast<AudioSources::enAudioSources>(poFgSrc->sGetId().enSourceClass))
                        , poFgSrc->sGetId().u16SubSource
                        , poFgSrc->sGetId().u16SubSource
                        , poNextSource->sGetId()));
            poFgSrc->vOff(poNextSource->sGetId());
         }else{
            //Maybe we do not have any source
            poFgSrc->vOff(SourceID(AudioSources::NONE, 0));
         }

      }

      m_pcoAudioStack->trace();
      bResult = TRUE;
   }
   else
   {
      ETG_TRACE_ERR(("NULL == poFgSrc or m_pcoAudioStack == NULL"));
      bResult = FALSE;
   }
   return bResult;
}

/*!************************************************************************
 * METHOD:        bMIXOFF
 * CLASS:         clAudioSourceController
 * DESCRIPTION:   Use this Interface Method to deativate a Mix source
 *                Sequence is:
 *                - Stop this source (mute-ramp)
 *                - Wait for asynchronous confirmation of stop(mute-ramp finished)
 *                - Send Broadcast Notification (Source stopped)
 *                This method initiates this sequence by calling vOff()
 *
 * PARAMETER:     u8MixSource
 *                u32UserData          reserved
 * RETURNVALUE:   bSucess
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tBool clAudioSourceController::bMIXOFF(SourceID mixSrc, tU32 u32UserData)
{
   tBool bResult = FALSE;
   clAudioSource* poMixSrc = NULL;

   poMixSrc = clAudioSourceFactory::getAudioSource(mixSrc);

   if (NULL != poMixSrc && m_pcoAudioStack != NULL)
   {
      //trace("bMIXOFF(%s)", poMixSrc->pacGetName());
    ETG_TRACE_USR2(("bMIXOFF SourceClass: %d subID %#x(%d)"
            ,ETG_CENUM(AudioSources::enAudioSources,static_cast<AudioSources::enAudioSources>(poMixSrc->sGetId().enSourceClass))
            , poMixSrc->sGetId().u16SubSource
            , poMixSrc->sGetId().u16SubSource));

      // a mix source can directly be stopped. The mute-demute-phases are skipped.
      // No other sources have to be started after this stop.
      // No Crossover and no BG-Exchange-Triggers are sent

      if (u32UserData & BITMASK_FORCE_OFF) // force OFF
      {
        ETG_TRACE_USR1(("bMIXOFF(): SourceClass: %d subID %#x(%d)"
            , ETG_CENUM(AudioSources::enAudioSources,static_cast<AudioSources::enAudioSources>(mixSrc.enSourceClass))
            , mixSrc.u16SubSource
            , mixSrc.u16SubSource));
        poMixSrc->vResetPlayCmdCounter();
      }
      poMixSrc->vOff(poMixSrc->sGetId());

      //m_pcoAudioStack->trace();
      bResult = TRUE;
   }
   else
   {
      ETG_TRACE_ERR(("Null Pointer for Mix source or AudioStack"));
      bResult = FALSE;
   }
  return bResult;
}


/*!************************************************************************
 * METHOD:        vCrossOver
 * CLASS:         clAudioSourceController
 * DESCRIPTION:   The trigger for animation sync
 *
 *
 * PARAMETER:
 * RETURNVALUE:
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tVoid clAudioSourceController::vCrossOver()
{
   //notify observers
   //trace("--- CROSSOVER TRIGGER ---");
   ETG_TRACE_USR4(("vCrossOver"));
}




/*!************************************************************************
 * METHOD:        vOff_done
 * CLASS:         clAudioSourceController
 * DESCRIPTION:   Callback from AudioSource
 *                The AudioSource indicates, that the mute-ramp was finished.
 *                AudioSources send this acknowledge also, if no mute-ramp
 *                was performed. E.g. when an off() is requested while the
 *                AudioSource is already in off-state.
 *
 *                When this confirmation arrives, the source is considered
 *                to be sopped completely, so send the stop-notification now.
 *
 * PARAMETER:     src   The AudioSource which has finished the mute-ramp
 * RETURNVALUE:
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tVoid clAudioSourceController::vOff_done(clAudioSource* src)
{
   clAudioSource* topOfStackSrc = NULL;
   clAudioSource* currentActiveSrc = NULL;
   clAudioSource* topMixSource = NULL;

   if (src != NULL && m_pcoAudioStack != NULL)
   {
      // stopped sources are removed from the stack
      m_pcoAudioStack->RemoveSource(*src);

      ETG_TRACE_USR2(("vOff_done: SourceClass: %d subID %#x(%d)"
            ,ETG_CENUM(AudioSources::enAudioSources,static_cast<AudioSources::enAudioSources>(src->sGetId().enSourceClass))
            , src->sGetId().u16SubSource
            , src->sGetId().u16SubSource));

      if (m_poObserver != NULL)
      {
         m_poObserver->vAudioSourceStopped(src->sGetId());
      }

      topOfStackSrc = pcoGetTopOfStackAudiosource();
      currentActiveSrc = pcoGetCurrentActiveAudiosource();

      if (src->getType() == clStackRules::typeBg)
      {
         clAudioSource* currentBGsrc = m_pcoAudioStack->GetCurrentBgSource();

         if (currentBGsrc != NULL && m_poObserver != NULL)
         {
            if (currentBGsrc == topOfStackSrc)
            {
               //foreground exchange
               ETG_TRACE_USR4(("vOff_done: FOREGROUND exchange of entertainment source: SourceClass: %d subID %#x(%d)"
               , ETG_CENUM(AudioSources::enAudioSources,static_cast<AudioSources::enAudioSources>(currentBGsrc->sGetId().enSourceClass))
               , currentBGsrc->sGetId().u16SubSource
               , currentBGsrc->sGetId().u16SubSource));
               m_poObserver->vEntertainmentSourceExchanged(currentBGsrc->sGetId(), FALSE);
            }
            else
            {
               //background exchange
               // bring new bg src from off to pause
               currentBGsrc->vPause(currentBGsrc->sGetId());
               ETG_TRACE_USR4(("vOff_done: BACKGROUND exchange of entertainment SourceClass: %d subID %#x(%d)"
               , ETG_CENUM(AudioSources::enAudioSources,static_cast<AudioSources::enAudioSources>(currentBGsrc->sGetId().enSourceClass))
               , currentBGsrc->sGetId().u16SubSource
               , currentBGsrc->sGetId().u16SubSource));
               m_poObserver->vEntertainmentSourceExchanged(currentBGsrc->sGetId(), TRUE);
            }
         }
      }
//      else
//         if (src->getType() == clStackRules::typeFg)
//         {
//
//         }

      //Take care of Mix-Sources that got rule WaitFor()
      //to wait for this Source
      topMixSource = pcoGetTopOfStackMixsource();
      if (topMixSource != NULL)
      {
        ETG_TRACE_USR4(("vOff_done,topMixSource is not NULL"));
         if ((WaitForSourceNeeded(*topMixSource)))
         {
           ETG_TRACE_USR4(("vOff_done,vWaitFor"));
            topMixSource->vWaitFor();
         }
         else
         {
            //We can proceed directly
//            if(topMixSource->pclGetState() != clSrcStateFactory::pclCreateOn())
//            {
           ETG_TRACE_USR4(("vOff_done,vOn"));
               topMixSource->vOn();
//            }
         }
      }

      if (currentActiveSrc == NULL && topOfStackSrc != NULL)
      {
        ETG_TRACE_USR4(("vOff_done,currentActiveSrc is NULL and topOfStackSrc is not NULL"));
         if (WaitForSourceNeeded(*topOfStackSrc))
         {
           ETG_TRACE_USR4(("vOff_done,vWaitFor"));
            topOfStackSrc->vWaitFor();
         }
         else
         {
            //We can proceed directly
           ETG_TRACE_USR4(("vOff_done,vOn"));
            topOfStackSrc->vOn();
         }
      }
      m_pcoAudioStack->trace();
   }
}

/*!************************************************************************
 * METHOD:        vPause_done
 * CLASS:         clAudioSourceController
 * DESCRIPTION:   Callback from AudioSource
 *                The AudioSource indicates, that the mute-ramp was finished.
 *                AudioSources send this acknowledge also, if no mute-ramp
 *                was performed. E.g. when an pause() is requested while the
 *                AudioSource is already in pause-state.
 *
 *                When this confirmation arrives, the source is considered
 *                to be stopped completely, so send the stop-notification now.
 *
 * PARAMETER:     src   The AudioSource which has finished the mute-ramp
 * RETURNVALUE:
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tVoid clAudioSourceController::vPause_done(clAudioSource* src)
{
  clAudioSource* topOfStackSrc = NULL;
  clAudioSource* currentActiveSrc = NULL;

  if(src != NULL)
  {
    // paused sources are NOT removed from the stack
    // trace("---Notification: AudioSource Stopped: %s", src->pacGetName());

    ETG_TRACE_USR2(("vPause_done: SourceClass: %d subID %#x(%d)"
            ,ETG_CENUM(AudioSources::enAudioSources,static_cast<AudioSources::enAudioSources>(src->sGetId().enSourceClass))
            , src->sGetId().u16SubSource
            , src->sGetId().u16SubSource));
    if (m_poObserver != NULL)
    {
      m_poObserver->vAudioSourceStopped(src->sGetId());
    }

    topOfStackSrc    = pcoGetTopOfStackAudiosource();
    currentActiveSrc = pcoGetCurrentActiveAudiosource();

    if (currentActiveSrc == NULL && topOfStackSrc != NULL)
    {
      //Check for PushAction WaitFor rule
      if (WaitForSourceNeeded(*topOfStackSrc))
      {
        topOfStackSrc->vWaitFor();
      }
      else
      {
        //We can proceed directly
        topOfStackSrc->vOn();
      }
    }
    m_pcoAudioStack->trace();
  }
}

/*!************************************************************************
 * METHOD:        vOn_done
 * CLASS:         clAudioSourceController
 * DESCRIPTION:   Callback from AudioSource
 *                The AudioSource indicates, that the demute-ramp was finished.
 *                AudioSources send this acknowledge also, if no demute-ramp
 *                was performed. E.g. when an on() is requested while the
 *                AudioSource is already in on-state.
 *
 *                When this confirmation arrives, the source is considered
 *                to be started completely, so send the start-notification now.
 *
 *
 * PARAMETER:     src   The AudioSource which has finished the demute-ramp
 * RETURNVALUE:
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tVoid clAudioSourceController::vOn_done(clAudioSource* src)
{
  if (src != NULL && src->sGetId().enSourceClass != AudioSources::NONE)
  {
    //trace("---Notification: AudioSource Started: %s", src->pacGetName());
    ETG_TRACE_USR2(("vOn_done: SourceClass: %d subID %#x(%d)"
            ,ETG_CENUM(AudioSources::enAudioSources,static_cast<AudioSources::enAudioSources>(src->sGetId().enSourceClass))
            , src->sGetId().u16SubSource
            , src->sGetId().u16SubSource));
    if (m_poObserver != NULL)
    {
      m_poObserver->vAudioSourceStarted(src->sGetId());
    }

  }

#ifdef PAUSE_MIX_SOURCES
  tBool bTopOfSrc = FALSE;
  clAudioSource* topOfStackSrc = pcoGetTopOfStackAudiosource();
  clAudioSource* poActiveMixSrc = pcoGetTopOfStackMixsource(src);

  if( poActiveMixSrc != NULL)
  {
    // play a mix source which is higher than the new FgSrc
    if(poActiveMixSrc != NULL)
    {
      if (poActiveMixSrc->pclGetState() == clSrcStateFactory::pclCreateOff()
      || poActiveMixSrc->pclGetState() == clSrcStateFactory::pclCreatePause()
      || poActiveMixSrc->pclGetState() == clSrcStateFactory::pclCreateRampDownToOff()
      || poActiveMixSrc->pclGetState() == clSrcStateFactory::pclCreateRampDownToPause())
      {
        vTrace((tU16)TRACE_AUDSM_SRC_MIXON_AFTER_BG_ONDONE,poActiveMixSrc->u8GetId());
        poActiveMixSrc->vOn();
      }
    }
  }
#endif

  if (m_pcoAudioStack != NULL)
  {
    m_pcoAudioStack->trace();
  }
}

/*!************************************************************************
 * METHOD:        vInit_started
 * CLASS:         clAudioSourceController
 * DESCRIPTION:   Callback from AudioSource
 *                The AudioSource indicates, that the init-state has been
 *                entered.
 *
 *                When this confirmation arrives, the source is considered
 *                to be about to pepare for play-request
 *
 * PARAMETER:     src   The concerned AudioSource
 * RETURNVALUE:
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tVoid clAudioSourceController::vInit_started(clAudioSource* src)
{
  if (src != NULL && src->sGetId().enSourceClass != AudioSources::NONE)
  {
    ETG_TRACE_USR2(("vInit_started: SourceClass: %d subID %#x(%d)"
            ,ETG_CENUM(AudioSources::enAudioSources,static_cast<AudioSources::enAudioSources>(src->sGetId().enSourceClass))
            , src->sGetId().u16SubSource
            , src->sGetId().u16SubSource));
    if (m_poObserver != NULL)
    {
      m_poObserver->vAudioSourceInitStarted(src->sGetId());
    }
  }

  if (m_pcoAudioStack != NULL)
  {
    m_pcoAudioStack->trace();
  }
}


/*!************************************************************************
 * METHOD:        vResetStack
 * CLASS:         clAudioSourceController
 * DESCRIPTION:   Interface method to clean up the stack the "hard way"
 *                The Stack is completely cleaned up. All active sources are
 *                stopped and removed from the stack.
 *
 * PARAMETER:
 * RETURNVALUE:
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tVoid clAudioSourceController::vResetStack()
{
   if (NULL != m_pcoAudioStack)
   {
      ETG_TRACE_USR2(("vResetStack"));
      clAudioSource *pcoAudioSource = m_pcoAudioStack->GetTopSource();

      // ensure silence during stack-reset
      bON(SourceID(AudioSources::MUTE_SYSTEM,0), 0);

      while(pcoAudioSource)
      {
         // prevent the system-mute to be stopped, because this could demute the audio !
         if (pcoAudioSource->sGetId().enSourceClass != AudioSources::MUTE_SYSTEM)
         {
            pcoAudioSource->vOff(pcoAudioSource->sGetId());
         }
         pcoAudioSource = m_pcoAudioStack->GetUnderlayingSource(*pcoAudioSource);
      }
      m_pcoAudioStack->vResetStack();
      clAudioSourceFactory::resetAudioSources();
      bSELECT( SourceID(AudioSources::NONE, 0));
      bON(     SourceID(AudioSources::MUTE_LIMITED_SYSTEM, 0));
      bON(     SourceID(AudioSources::MUTE_SYSTEM, 0));
   }
}

tVoid clAudioSourceController::vTraceStack() const
{
   if (m_pcoAudioStack != NULL)
   {
      m_pcoAudioStack->trace();
   }
}



//returns AudioSource which is top of stack, except Mix sources
clAudioSource* clAudioSourceController::pcoGetTopOfStackAudiosource()
{
   clAudioSource *pcoAudioSourceIterator = NULL;

   if (NULL != m_pcoAudioStack)
   {
      pcoAudioSourceIterator = m_pcoAudioStack->GetTopSource();

      while(pcoAudioSourceIterator && pcoAudioSourceIterator->getType() == clStackRules::typeMix)
      {
            pcoAudioSourceIterator = m_pcoAudioStack->GetUnderlayingSource(*pcoAudioSourceIterator);
      }
   }
   return pcoAudioSourceIterator;
}

/**
 * @brief returns the AudioSource on stack which is not Mix AND NOT in Pause/Off
 * there is always exactly one result
 */
clAudioSource* clAudioSourceController::pcoGetCurrentActiveAudiosource()
{
   clAudioSource *pcoAudioSourceIterator = NULL;

   if (NULL != m_pcoAudioStack)
   {
      pcoAudioSourceIterator = m_pcoAudioStack->GetTopSource();

      while(pcoAudioSourceIterator && (pcoAudioSourceIterator->getType() == clStackRules::typeMix
         || !(pcoAudioSourceIterator->bIsActive())) )
      {
            pcoAudioSourceIterator = m_pcoAudioStack->GetUnderlayingSource(*pcoAudioSourceIterator);
      }
   }
   return pcoAudioSourceIterator;
}

/**
 * @brief returns the true/false if source is current active audio source
 *
 */
tBool clAudioSourceController::bIsCurrentActiveAudiosource(SourceID srcID)
{
  bool bResult = false;
  clAudioSource *pcoAudioSourceIterator = pcoGetCurrentActiveAudiosource();
  if((NULL != pcoAudioSourceIterator) &&
        (srcID == pcoAudioSourceIterator->sGetId()))
  {
    bResult = true;
  }
  return bResult;
}

/**
 * @brief returns the AudioSource on stack which is Mix AND NOT in Pause/Off
 * there is always exactly one result
 * @return NULL if there is no Mix Source active, else Pointer to active MixSource
 */
clAudioSource* clAudioSourceController::pcoGetCurrentMixSource()
{
   clAudioSource *pcoAudioSourceIterator = NULL;

   if (NULL != m_pcoAudioStack)
   {
      pcoAudioSourceIterator = m_pcoAudioStack->GetTopSource();

      while(pcoAudioSourceIterator && (pcoAudioSourceIterator->getType() != clStackRules::typeMix
         || !(pcoAudioSourceIterator->bIsActive())) )
      {
         pcoAudioSourceIterator = m_pcoAudioStack->GetUnderlayingSource(*pcoAudioSourceIterator);
      }
   }
   return pcoAudioSourceIterator;
}


/**
 * @brief most top MixAudioSource of the Stack
 * @return NULL if there is no Mix source on stack
 */
clAudioSource* clAudioSourceController::pcoGetTopOfStackMixsource()
{
   clAudioSource *pcoAudioSourceIterator = NULL;
   if (NULL != m_pcoAudioStack)
   {
      pcoAudioSourceIterator = m_pcoAudioStack->GetTopSource();
      while(pcoAudioSourceIterator && (pcoAudioSourceIterator->getType() != clStackRules::typeMix))
      {
          pcoAudioSourceIterator = m_pcoAudioStack->GetUnderlayingSource(*pcoAudioSourceIterator);
      }

      if(pcoAudioSourceIterator && (pcoAudioSourceIterator->getType() == clStackRules::typeMix))
      {
         return pcoAudioSourceIterator;
      }
   }
   return NULL;
}

/**
 * @briefCopies pointer AudioSources that are actually on Stack
 * to given List.
 * @param sourceList
 * @return
 */
tVoid clAudioSourceController::vGetStackList(std::list<  clAudioSource * > &sourceList)
{
   //Get TopSource
   clAudioSource *pcoAudioSource = m_pcoAudioStack->GetTopSource();
   if(pcoAudioSource != NULL)
   {
      ETG_TRACE_ERR(("vGetStackList: Top SourceClass: %d subID %#x(%d)"
         ,ETG_CENUM(AudioSources::enAudioSources,static_cast<AudioSources::enAudioSources>(pcoAudioSource->sGetId().enSourceClass))
         , (tU16)pcoAudioSource->sGetId().u16SubSource
         , (tU16)pcoAudioSource->sGetId().u16SubSource));
   }else{
      ETG_TRACE_ERR(("vGetStackList: TopSource is NULL"));
   }
   //Add every source of the Stack
   while(pcoAudioSource)
   {
      if(pcoAudioSource != NULL)
      {
      ETG_TRACE_ERR(("vGetStackList: Push SourceClass: %d subID %#x(%d)"
         ,ETG_CENUM(AudioSources::enAudioSources,static_cast<AudioSources::enAudioSources>(pcoAudioSource->sGetId().enSourceClass))
         , (tU16)pcoAudioSource->sGetId().u16SubSource
         , (tU16)pcoAudioSource->sGetId().u16SubSource));
      }
      sourceList.push_back(pcoAudioSource);
      pcoAudioSource = m_pcoAudioStack->GetUnderlayingSource(*pcoAudioSource);
   }
}


/*!************************************************************************
 * METHOD:        bIsAudioMuted
 * CLASS:         clAudioSourceController
 * DESCRIPTION:   specifies, if the audio is currently muted
 *                This method return true, if (and only if) one of the virtual
 *                mute sources: [SYSTEM_MUTE, PIN_MUTE, ENTERTAINMENT_MUTE
 *                LIMITED_SYSTEM_MUTE]
 *                is on top of stack.
 *                This method does not take care of temp. Phone-,
 *                Cluster- and Diagnosis-Mute
 *                If a virtual Mute-Source is on top of stack, this method
 *                returns TRUE, no matter if the AudioSource, which was active
 *                before, is already completely muted. (Mute-Ramp running)
 * PARAMETER:
 * RETURNVALUE:
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
//tBool clAudioSourceController::bIsAudioMuted()
//{
//   // check for top of stack audio source (excluding MIX sources)
//   clAudioSource *pcoAudioSource = pcoGetTopOfStackAudiosource();
//   if (pcoAudioSource != NULL)
//   {
//      if(pcoAudioSource->getGroup() == clStackRules::group_t::groupInternal)
//      {
//         return TRUE;
//      }
//   }
//   return FALSE;
//}

/*!************************************************************************
 * METHOD:        bIsEntertainmentSourcePaused
 * CLASS:         clAudioSourceController
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tBool clAudioSourceController::bIsEntertainmentSourcePaused()
{
   tBool bResult = FALSE;

   // check for top of stack audio source (excluding MIX sources)
   clAudioSource *pcoAudioSource = pcoGetTopOfStackAudiosource();
   if (pcoAudioSource != NULL)
   {
      if (pcoAudioSource->getType() != clStackRules::typeBg)
      {
         // there is a None-Entertainment-Source on top of stack, so
         // the entertainment source is muted
         bResult = TRUE;
      }
   }
   return bResult;
}



#if 0
/*!************************************************************************
 * METHOD:        u8GetActiveAudioComponent
 * CLASS:         clAudioSourceController
 * DESCRIPTION:   returns the current audiocomponent
 *                IMPORTANT NOTE:
 *                1) This method returns the asynchronous component!
 *                This means, that during a running source switch (mute-demute-ramp)
 *                the currently active source is returned, taking care of mute-phases.
 *                OBSOLETE:
 *                1) This method returns the synchronous component!
 *                This means, that during a running source switch (mute-demute-ramp)
 *                already the new source is returned, even if the previous
 *                source is still in mute-ramp.
 *                END OBSOLETE
 *                2) This method refers to valid audio components only.
 *                This means, that if there are virtual sources  on top of the
 *                stack (e.g. System-Mute, Entertainment-Mute, etc), they
 *                will be ignored by this method!
 *
 *
 * PARAMETER:
 * RETURNVALUE:   [AudioComponent::NAVI, AudioComponent::PHONE,
 *                 AudioComponent::RADIO_ANNOUNCEMENT, AudioComponent::AUDIO]
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/

tU8 clAudioSourceController::u8GetActiveAudioComponent()
{
   clAudioSource *pcoAudioSourceIterator = NULL;
   tU8 u8ActiveAudioComponent = AudioComponent::AC_AUDIO;
   tU8 u8CurrentAudioSource = AudioSources::NONE;

   if (NULL != m_pcoAudioStack)
   {
      pcoAudioSourceIterator = m_pcoAudioStack->GetTopSource();

      while(pcoAudioSourceIterator)
      {
         u8CurrentAudioSource = pcoAudioSourceIterator->u8GetId();
         /* exclude rampDown Phases if you require synchronous behaviour
         if (pcoAudioSourceIterator->pclGetState() != clSrcStateFactory::pclCreateRampDownToPause()
            && pcoAudioSourceIterator->pclGetState() != clSrcStateFactory::pclCreateRampDownToOff())
            */
         {
            if (u8CurrentAudioSource == AudioSources::NAVISPEECH )
            {
               u8ActiveAudioComponent = AudioComponent::AC_NAVI;
               break;
            }
            else if (u8CurrentAudioSource == AudioSources::TELEPHONE)
            {
               u8ActiveAudioComponent = AudioComponent::AC_PHONE;
               break;
            }
            else if (u8CurrentAudioSource == AudioSources::RADIO_TA)
            {
               u8ActiveAudioComponent = AudioComponent::AC_RADIO_ANNOUNCEMENT;
               break;
            }
            else if (pcoAudioSourceIterator->getType() == clStackRules::typeBg)
            {
               u8ActiveAudioComponent = AudioComponent::AC_AUDIO;
               break;
            }
         }

         // iterator points to a source which is not of a valid AudioComponent type, so continues
         // iterations going down the stack
         pcoAudioSourceIterator = m_pcoAudioStack->GetUnderlayingSource(*pcoAudioSourceIterator);
      }
   }
   return u8ActiveAudioComponent;
}
#endif

/**
 * Check if given source is allowed to Enter Stack
 * @param poAudioSource stack is checked against this source
 * @return
 */
clStackRules::allow_t clAudioSourceController::enCheckSource(clAudioSource* poAudioSource)
{
   if(m_pcoAudioStack)
      if( m_pcoAudioStack->AllowSource(*poAudioSource) != NULL)
         return clStackRules::allowed;
   else
      return clStackRules::disallowed; // only for lint, case will never happen
}

/**
 * Retrieves most top exclusive Audiosource ID
 * @return if the stack is empty NONE is returned
 */
SourceID clAudioSourceController::u8GetTopOfStackAudiosource()
{
   clAudioSource* poTopStackSrc = NULL;
   SourceID  sTopStackSrc (AudioSources::NONE,0);

   poTopStackSrc = pcoGetTopOfStackAudiosource();
   if(poTopStackSrc != NULL)
   {
      sTopStackSrc =  poTopStackSrc->sGetId();
   }
   return sTopStackSrc;
}

tVoid clAudioSourceController::vHandleSrcActError(SourceID errSrc)
{
   //Get errSrc Obj
   clAudioSource* pErrSrc = clAudioSourceFactory::getAudioSource(errSrc);

   //Iterate through list and find Source in RampUp/Down state
   if(pErrSrc != NULL)
   {
      clSrcState* state = pErrSrc->pclGetState();
      ETG_TRACE_ERR(("vHandleSrcActError:  SourceClass: %d subID %#x(%d), State: %d"
            ,ETG_CENUM(AudioSources::enAudioSources,static_cast<AudioSources::enAudioSources>(errSrc.enSourceClass))
            , errSrc.u16SubSource
            , errSrc.u16SubSource
            , ETG_CENUM(AudioStates::enAudioStates,   pErrSrc->pclGetState()->enGetStateID())));
      //Reset Source
      ETG_TRACE_ERR(("vHandleSrcActError: Reset source"));
      pErrSrc->vReset();
      // Off_Done is not called implicitly when reset the source
      pErrSrc->vNotifyOffDone();
   }
}

} //namespace

